\name{rTableICC.RxC}
\alias{rTableICC.RxC}

\title{
Randomly Generate RxC Contingency Tables over Intraclass-Correlated Individuals
}
\description{
A generic function that generates R x C contingency tables over intraclass-correlated cells under product multinomial, multinomial or Poisson sampling plans. 
}
\usage{
rTableICC.RxC(p=NULL,theta,M,row.margins=NULL,col.margins=NULL,sampling="Multinomial",
               N=1,lambda=NULL,zero.clusters=FALSE,print.regular=TRUE,print.raw=FALSE)

}

\arguments{
\item{p}{
  A finite \eqn{R\times C} matrix of cell probabilities.
}
  \item{theta}{
  A finite and positive valued \eqn{(T-1)\times 1} vector of predetermined ICCs, where \eqn{T} is the maximum number of individuals allowed in each cluster. The first element of \code{theta} represents the ICC for clusters of size 2. 
  }
  \item{M}{
  The total number of clusters under each factor. It must be a positive scalar under multinomial and Poisson sampling plans and can be a vector of length equal to that of number of rows or columns under product multinomial sampling plan. If it is a scalar, number of clusters under levels of fixed margin is assigned equal.
  }  
  \item{row.margins}{
   Includes fixed row margins under product multinomial sampling plan and not required for both multinomial and Poisson sampling plans. 
  }    
  \item{col.margins}{
  Includes fixed column margins under product multinomial sampling plan and not required for both multinomial and Poisson sampling plans.
  }  
  \item{sampling}{
  Sampling plan. It takes 'Product' for product multinomial sampling, 'Multinomial' for multinomial sampling, and 'Poisson' for Poisson sampling plans.
  }  
  \item{N}{
  Total number of individuals to be generated under product multinomial or multinomial sampling plans. It is a positive integer of total sample size under multinomial sampling plan and not required under both product multinomial and Poisson sampling plans. If \code{N} is not a positive integer, it is converted to do so.
  }  
  \item{lambda}{
  Mean number of individuals in each cell of table. It is an \eqn{R\times C} positive matrix under Poisson sampling plan and not required for both multinomial and product multinomial sampling plans.
  }  
  \item{zero.clusters}{
  If \code{TRUE}, generation of clusters with no individuals are allowed. Otherwise, each cluster has at least one individual. 
  }
  \item{print.regular}{
  If \code{TRUE}, generated random table is printed in \eqn{2\times 2\times K} format. Otherwise, generated random table is printed in two dimensional format. 
  }  
  \item{print.raw}{
  If \code{TRUE}, generated raw data is printed on the screen.
  }   
}
\details{
To generate random tables under multinomial sampling plan, first total sample size is distributed to clusters with equal probabilities using the code \code{rmultinom(1, N, rep(1/M,M))}. Then the package \code{partitions} is utilized to distribute individuals across cells under the pre-determined intraclass correlations. Let \eqn{n} and \eqn{m} be integer to be partitioned (cluster size) and order of partition, respectively. If there is more than one individual (\eqn{n>1}) in a cluster, all possible compositions of order \eqn{RC} of cluster size \eqn{n} into at most \eqn{m} parts are generated by using \code{compositions} function. This provides all possible distributions of individuals in the cluster of interest into \eqn{RC} cells. If all individuals are at the same cell, the following equation is used to calculate the probability that all individuals in the \eqn{i}th cluster fall in the same cell of a contingency table of interest:

\deqn{
      \theta_{t}p_{ij}+(1-\theta_{t})(p_{ij})^{t},
} 
      
where \eqn{i=1,\dots,R}, \eqn{j=1,\dots,C}, \eqn{0\le\theta\le 1}, \eqn{\theta_{t}} is the intraclass correlation for clusters of size \eqn{t} for \eqn{t=2,\dots,T}, and \eqn{\theta_{1}=0}. Otherwise, the probability that the individuals are in different but specified cells is calculated as follows:

\deqn{
      (1-\theta_{t})\prod_{ij}(p_{ij})^{n_{kij}},
}

where \eqn{n_{kij}} be the number of individuals from \eqn{k}th cluster falling in the \eqn{i}th row and \eqn{j}th column of the considered RxC table (Altham, 1976; Nandram and Choi, 2006; Demirhan, 2013). This provides probability of each possible distribution. Then, calculated probabilities are normalized and the function \code{rDiscrete} is utilized to randomly select one of the generated compositions. By this way, a realization is obtained for each cluster having more than one individual. If there is only one individual in a cluster, a realization is obtained by assigning the individual to cells according to the entered cell probabilities using the function \code{rDiscrete}. The resulting random RxC table is constructed by combining these realizations. 

To generate random tables under product multinomial sampling plan, at least one of \code{row.margins} or \code{col.margins} must be entered. Because both cell probabilities and fixed row or column margins are entered at the same time, margin probabilities calculated over fixed margins and entered \eqn{R\times C} matrix of cell probabilities must be equal to each other. To ensure intraclass correlations, the same manner as multinomial sampling plan is applied to the fixed margin. Suppose that row totals are fixed and \eqn{n_{i+}} denotes fixed row margins. Then with the counts satisfying \eqn{\sum_{j}n_{ij}=n_{i+}}, we have the following multinomial form that \code{rTableICC.RxC} uses (Agresti, 2002):

\deqn{
      \frac{n_{i+}!}{\prod_{j}n_{ij}!}\prod_{j}p_{j|i}^{n_{ij}},
      } 

where \eqn{j=1,\dots,C}, \eqn{n_{ij}} is the count of cell \eqn{(i,j)}, and given that an individual is in \eqn{i}th row, \eqn{p_{j|i}} is the conditional probability of being in the \eqn{j}th column of table. This multinomial form is used to generate data under each row margin. When column totals are fixed the same manner as the case of fixed row totals is followed.
  
To generate random tables under Poisson sampling plan, the same manner as multinomial sampling plan is taken except cell counts are generated from Poisson distribution with entered mean cell counts and total sample size is calculated over the generated cell counts. If zero sized clusters are not allowed, truncated Poisson distribution is used to generate cluster counts.

Because total sample size is randomly distributed into the clusters, it is coincidentally possible to have clusters with more individuals than the allowed maximum cluster size. In this case, the following error message is generated: 

\code{Maximum number of individuals in one of the clusters is 14, which is greater than maximum allowed cluster size. (1) Re-run the function,
(2) increase maximum allowed cluster size by increasing the number of elements of theta, (3) increase total number of clusters, or (4) decrease total number of individuals!} 

and execution is stopped.  
}
\value{
Let \eqn{C} be the set of clusters in which all individuals fall in a single cell of the contingency table and \eqn{C'} be the complement of \eqn{C},  \eqn{K} be the number of centers, and \eqn{T} be the maximum cluster size.
 
 A list with the following elements is generated:
\item{g.t}{A \eqn{R\times C\times (T-1)} dimensional array including the number of clusters of size t in \eqn{C'} of size \eqn{t} with all individuals in cell \eqn{(i,j)}, where \eqn{i=1,\dots,R}, \eqn{j=1,\dots,C}, and \eqn{t=2,\dots,T}.}
\item{g.tilde}{A \eqn{(T-1)\times 1} dimensional vector including the number of clusters of size \eqn{t} in \eqn{C'}, where \eqn{t=2,\dots,T}.}
\item{rTable}{A \eqn{RC\times 1} dimensional vector including generated RxC contingency table in a row.}
\item{rTable.raw}{Generated table in a \eqn{R\times C\times N} dimensional array in raw data format.}
\item{rTable.regular}{Generated table in an \eqn{R\times C} dimensional matrix.}
\item{N}{Total number of generated individuals.}
\item{cluster.size}{Size of each generated cluster.}
\item{sampling}{Used sampling plan in data generation.}
\item{M}{Total number of clusters.}
\item{R}{Number of rows.}
\item{C}{Number of columns.}
\item{T}{Maximum allowed cluster size.}
\item{ICC}{Returns \code{TRUE} to indicate the data is generated under intraclass-correlated clusters.}
\item{structure}{Returns "R x C" to indicate structure of generated table is "R x C."}
\item{print.raw}{\code{TRUE} if generated table will be printed in raw data format.}
\item{print.regular}{\code{TRUE} if generated table will be printed in the format determined by \code{structure}.}
}
\references{
Agresti A. (2002) \emph{Categorical Data Analysis}, Wiley, New York.

Altham, P.M. (1976) Discrete variable analysis for individuals grouped into families, \emph{Biometrika} \bold{63}, 263--269.

Nandram, B. and Choi, J.W. (2006) Bayesian analysis of a two-way categorical table incorporating intraclass correlation,
\emph{Journal of Statistical Computation and Simulation} \bold{76}, 233--249.

Demirhan, H. (2013) Bayesian estimation of log odds ratios over two-way contingency tables with intraclass-correlated cells, \emph{Journal of Applied Statistics} \bold{40}, 2303--2316.

Demirhan, H. and Hamurkaroglu, C. (2008) Bayesian estimation of log odds ratios from RxC and 2 x 2 x K contingency tables, \emph{Statistica Neerlandica} \bold{62}, 405--424.
}
\author{
Haydar Demirhan

Maintainer: Haydar Demirhan <haydar.demirhan@rmit.edu.au>
}

\examples{
# --- Generate a 2x3 contingency table under multinomial sampling plan ---
max.cluster.size=9                          # Maximum allowed cluster size
num.cluster=12                              # Total number of clusters 
ICCs=array(0.1,dim=max.cluster.size)        # Assign equal ICCs for this exmaple
ICCs[1]=0                                   # Assign zero ICC to clusters with 
                                            #  one individual 
sampl="Multinomial"                         # Generate table under multinomial 
                                            #  sampling plan
num.obs=24                                  # Number of observations to be 
                                            #  generated
cell.prob=array(1/6,dim=c(2,3))             # Cell probabilities sum up to one 
zeros=FALSE                                 # Do not allow zero sized clusters

x=rTableICC.RxC(p=cell.prob,theta=ICCs,M=num.cluster,sampling=sampl,
                 N=num.obs,zero.clusters=zeros,print.regular=TRUE,
                 print.raw=FALSE)
print(x) 

# --- Generate a 2x3 contingency table under product multinomial sampling plan ---
# --- with fixed row margins                                                   ---
sampl="Product"                             # Generate table under product 
                                            #  multinomial sampling plan
row=c(12,12)                                # Fixed row margins
cell.prob=array(0,dim=c(2,3))               # Cell probabilities sum up to one 
cell.prob[1,1:2]=0.2
cell.prob[1,3]=0.1
cell.prob[2,1:2]=0.1
cell.prob[2,3]=0.3                          # Marginal and cell probabilities 
                                            #  should be equal to each other

y1=rTableICC.RxC(p=cell.prob,theta=ICCs,row.margins=row,M=num.cluster,
                  sampling=sampl,print.regular=TRUE,print.raw=FALSE)
print(y1)

# --- Generate a 3x2 contingency table under product multinomial sampling plan ---
# --- with fixed cloumn margins                                                ---
col=c(12,12) 
cell.prob=array(0,dim=c(3,2))               # Cell probabilities sum up to one 
cell.prob[1:2,1]=0.2
cell.prob[1,2]=0.1
cell.prob[2,2]=0.1
cell.prob[3,1]=0.1
cell.prob[3,2]=0.3  

y2=rTableICC.RxC(p=cell.prob,theta=ICCs,col.margins=col,M=num.cluster,
                  sampling=sampl,print.regular=TRUE,print.raw=FALSE)
print(y2)

# --- Generate a 4x3 contingency table under Poisson sampling plan ---
sampl="Poisson"                             # Generate table under product 
                                            #  multinomial sampling plan
cell.prob=array(1/12,dim=c(4,3))            # Cell probabilities sum up to one 
cell.mean=array(4,dim=c(4,3))               # Define mean number of individuals 
                                            #  in each cell
max.cluster.size=19                         # Maximum allowed cluster size
ICCs=array(0.1,dim=max.cluster.size)        # Assign equal ICCs for this exmaple
ICCs[1]=0  

z=rTableICC.RxC(p=cell.prob,lambda=cell.mean,theta=ICCs,row.margins=row,
                 M=num.cluster,sampling=sampl,print.regular=TRUE,print.raw=FALSE)
print(z)

}
\keyword{ datagen }
\keyword{contingency}
\keyword{correlation}
\keyword{intraclass}
\keyword{Poisson}
\keyword{multinomial}
\keyword{product}