% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metaquant.R
\name{metaquant}
\alias{metaquant}
\title{Meta-Analysis of Quantiles and Functions of Quantiles}
\usage{
metaquant(
   data,
   method = "gld",
   effect.size.type = "median",
   opt = TRUE,
   single.family = FALSE,
   pool.studies = TRUE,
   ...)
}
\arguments{
\item{data}{a data frame with one row per study containing five-number
summaries and sample sizes.
For one-group studies, the input should contain the following columns:
\describe{
\item{\code{'min.g1'}}{minimum value}
\item{\code{'q1.g1'}}{first quartile}
\item{\code{'med.g1'}}{median}
\item{\code{'q3.g1'}}{third quartile}
\item{\code{'max.g1'}}{maximum value}
\item{\code{'n.g1'}}{sample size}
}
For two-group studies, also include the corresponding columns for the second
group: \code{min.g2}, \code{q1.g2}, \code{med.g2}, \code{q3.g2},
\code{max.g2}, and \code{n.g2}.}

\item{method}{character string specifying the density-based approach used to
perform the meta analysis of quantiles or their functions. Options:
\describe{
\item{\code{'gld'}}{The default option. Estimation method proposed by
De Livera et al. (2024) using the generalised lambda distribution (GLD).}
\item{\code{'qe'}}{Quantile Matching Estimation method proposed by McGrath
et al. (2020).}
}}

\item{effect.size.type}{character string specifying the quantile-based effect
size for the meta-analysis. Options:
\describe{
\item{\code{'median'}}{The default option. Median for single-group studies;
difference in medians for two-group studies.}
\item{\code{'q1'}}{First quartile for single-group studies; difference in
first quartiles for two-group studies.}
\item{\code{'q3'}}{Third quartile for single-group studies; difference in
third quartiles for two-group studies.}
\item{\code{'logr2'}}{Log ratio of squared IQRs between two groups. Only
applicable when the input data frame provides columns for both groups.}
}}

\item{opt}{logical; whether to apply the optimisation step of the
\code{"gld"} method when estimating its parameters. Default is \code{TRUE}.}

\item{single.family}{logical; for two-group studies using the \code{"qe"}
method, whether to assume the same parametric family of distributions for
both groups. Default is \code{FALSE}.
See \code{\link[metamedian:qe.study.level]{qe.study.level}}.}

\item{pool.studies}{logical; whether to pool study-specific effect sizes via
inverse-variance–weighted meta-analysis. Default is \code{TRUE}. If
\code{FALSE}, the function returns a list of effect sizes and their
within-study variance estimates. See \code{\link[metafor:rma.uni]{rma.uni}}.}

\item{...}{additional arguments passed to
\code{\link[metafor:rma.uni]{rma.uni}} for pooling.}
}
\value{
An object of class "rma.uni" or a list of effect sizes and their
estimated variances.
}
\description{
This function implements statistical methods for meta-analysis of quantiles
and functions of quantiles for single-group and two-group studies. The
function uses inverse-variance weighting to synthesise information from studies
that report five-number summaries (minimum, first quartile, median, third
quartile, maximum) and sample sizes—particularly useful for skewed outcomes.

The \code{metaquant} function currently supports two density-based frameworks:
(1) a Generalized Lambda Distribution (GLD) fitted via percentile matching,
following De Livera et al. (2024), to estimate parameters for meta-analysis of
medians and other quantiles; and (2) an extension of the Quantile Estimation
(QE) method of McGrath et al. (2020) to additional quantiles and functions of
quantiles, with derived standard errors for inverse-variance pooling.

The function facilitates meta-analyses of the following effect sizes:
\itemize{
\item Single-group quantiles: median (\eqn{m}), first quartile (\eqn{q_1}),
third quartile (\eqn{q_3}).
\item Two-group differences in quantiles: difference in medians
(\eqn{m_{g1}-m_{g2}}), difference in first quartiles
(\eqn{q_{1g1}-q_{1g2}}), difference in third quartiles
(\eqn{q_{3g1}-q_{3g2}}).
\item Ratio of squared interquartile ranges (IQRs) between two groups:
\eqn{r = (q_{3g1}-q_{1g1})^2 / (q_{3g2}-q_{1g2})^2}.
}

Portions of this implementation are adapted from
\code{\link[metamedian:metamedian]{metamedian}} for the QE
method, and have been extended to support other quantiles and functions of
quantiles beyond the median.
}
\examples{
# Example dataset of 5-number summaries (min, q1, med, q3, max) for 2 groups
data_2g <- data.frame(
  study.index = c("Study1", "Study2", "Study3"),
  min.g1 = c(15, 15, 13),
  q1.g1  = c(57, 59, 55),
  med.g1 = c(66, 68, 60),
  q3.g1  = c(74, 72, 69),
  max.g1 = c(108, 101, 100),
  n.g1   = c(226, 230, 200),
  min.g2 = c(18, 19, 15),
  q1.g2  = c(66, 71, 69),
  med.g2 = c(73, 82, 81),
  q3.g2  = c(80, 93, 89),
  max.g2 = c(110, 115, 100),
  n.g2   = c(226, 230, 200)
)
print(data_2g)

# Meta-analysis of difference in first quartiles
metaquant(data = data_2g, method = "gld", effect.size.type = "q1")
metaquant(data = data_2g, method = "qe",  effect.size.type = "q1")

# Meta-analysis of log ratio of squared IQRs
ma_lr <- metaquant(data = data_2g, method = "gld", effect.size.type = "logr2")
# Back-transform to original scale (ratio of squared IQRs)
est_r <- exp(ma_lr$b)                                       # pooled estimate
ci_r  <- exp(c(ma_lr$ci.lb, ma_lr$ci.ub))                   # confidence interval
pi_r  <- exp(c(predict(ma_lr)$pi.lb, predict(ma_lr)$pi.ub)) # prediction interval
est_r; ci_r; pi_r

}
\references{
De Livera, A. M., Prendergast, L., & Kumaranathunga, U. (2024). A novel
density-based approach for estimating unknown means, distribution visualisations and meta-analyses
of quantiles. \emph{arXiv preprint arXiv:2411.10971}. \url{https://arxiv.org/abs/2411.10971}.

King, R., Dean, B., Klinke, S., & van Staden, P. (2025). gld: Estimation
and use of the Generalised (Tukey) Lambda Distribution (R package Version 2.6.7). Comprehensive R
Archive Network (CRAN). https://doi.org/10.32614/CRAN.package.gld. \url{https://CRAN.R-project.org/package=gld}.

McGrath, S., Sohn, H., Steele, R., & Benedetti, A. (2020). Meta‐analysis of
the difference of medians. \emph{Biometrical Journal}, 62(1), 69-98.

McGrath, S., Zhao, X., Ozturk, O., Katzenschlager, S., Steele, R., & Benedetti, A. (2024).
Metamedian: an R package for meta‐analyzing studies reporting medians. \emph{Research Synthesis Methods}, 15(2), 332-346.
}
\seealso{
\code{\link[=est.mean]{est.mean()}}, \code{\link[=est.sd]{est.sd()}}
}
