% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/deviance_residuals.R
\name{cox_deviance_residuals}
\alias{cox_deviance_residuals}
\alias{cox_deviance_details}
\alias{cox_deviance_residuals_big}
\alias{cox_partial_deviance_big}
\alias{benchmark_deviance_residuals}
\title{Cox deviance residuals via C++ backends}
\usage{
cox_deviance_residuals(time, status, weights = NULL)

cox_deviance_details(time, status, weights = NULL)

cox_deviance_residuals_big(X, time_col, status_col, weights = NULL)

cox_partial_deviance_big(X, coef, time, status)

benchmark_deviance_residuals(time, status, iterations = 25, methods = list())
}
\arguments{
\item{time}{Numeric vector of follow-up times.}

\item{status}{Numeric or integer vector of the same length as \code{time} giving
the event indicators (1 for an event, 0 for censoring).}

\item{weights}{Optional non-negative case weights. When supplied they must
have the same length as \code{time}.}

\item{X}{A \code{\link[bigmemory:big.matrix-class]{bigmemory::big.matrix}} storing the
survival information column-wise.}

\item{time_col, status_col}{Integer indices pointing to the columns of \code{X}
that contain the follow-up time and event indicator respectively.}

\item{coef}{Numeric vector of regression coefficients used to evaluate the
partial log-likelihood and deviance on a \code{big.matrix} design.}

\item{iterations}{Number of iterations used by \code{\link[bench:mark]{bench::mark}} when
benchmarking the residual computations.}

\item{methods}{Optional named list of alternative residual implementations to
compare against in \code{\link{benchmark_deviance_residuals}}.}
}
\value{
\itemize{
\item \code{\link[=cox_deviance_residuals]{cox_deviance_residuals()}} and \code{\link[=cox_deviance_residuals_big]{cox_deviance_residuals_big()}} return a
numeric vector of deviance residuals.
\item \code{\link[=cox_deviance_details]{cox_deviance_details()}} returns a list with cumulative hazard,
martingale, and deviance residuals.
\item \code{\link[=cox_partial_deviance_big]{cox_partial_deviance_big()}} returns a list containing the partial
log-likelihood, deviance, and the evaluated linear predictor.
\item \code{\link[=benchmark_deviance_residuals]{benchmark_deviance_residuals()}} returns a \code{\link[tibble:tibble]{tibble::tibble}}.
}
}
\description{
Compute martingale and deviance residuals for Cox models without
materialising intermediate survival fits in R. The functions rely on
dedicated C++ implementations that operate either on in-memory vectors or on
\code{\link[bigmemory:big.matrix]{bigmemory::big.matrix}} objects to enable streaming computations on large
datasets.
}
\details{
\itemize{
\item \code{\link[=cox_deviance_residuals]{cox_deviance_residuals()}} operates on standard R vectors and matches the
output of \code{residuals(coxph(...), type = "deviance")} for right-censored
data without ties.
\item \code{\link[=cox_deviance_residuals_big]{cox_deviance_residuals_big()}} keeps the computation in C++ while reading
directly from a \code{big.matrix}, avoiding extra copies.
\item \code{\link[=cox_partial_deviance_big]{cox_partial_deviance_big()}} evaluates the partial log-likelihood and
deviance for a given coefficient vector and big design matrix. This is
useful when selecting the number of latent components via information
criteria.
}

\code{\link[=benchmark_deviance_residuals]{benchmark_deviance_residuals()}} compares the dedicated C++ implementation
against reference approaches (for example, the \code{survival} package) using
\code{\link[bench:mark]{bench::mark}}. The function returns a tibble with iteration statistics.
}
\examples{
if (requireNamespace("survival", quietly = TRUE)) {
  set.seed(123)
  time <- rexp(50)
  status <- rbinom(50, 1, 0.6)
  dr_cpp <- cox_deviance_residuals(time, status)
  dr_surv <- residuals(survival::coxph(survival::Surv(time, status) ~ 1),
                       type = "deviance")
  all.equal(unname(dr_cpp), unname(dr_surv), tolerance = 1e-6)
}

}
