#' @title Distributed Gao Principal Component Analysis
#' @description Performs distributed Gao-type principal component analysis
#' on a numeric dataset split across multiple nodes.
#'
#' @param data A numeric matrix containing the total dataset.
#' @param m An integer specifying the number of principal components for the first stage.
#' @param n1 An integer specifying the length of each data subset.
#' @param K An integer specifying the number of nodes.
#'
#' @return A list with the following components:
#' \describe{
#'   \item{AG1}{List of estimated loading matrices for the first-stage components for each node.}
#'   \item{AG2}{List of estimated loading matrices for the second-stage components for each node.}
#'   \item{DG3}{List of diagonal residual variance matrices for each node.}
#'   \item{sGhat}{List of covariance matrices of reconstructed data for each node.}
#' }
#'
#' @examples
#' set.seed(123)
#' data <- matrix(rnorm(500), nrow = 100, ncol = 5)
#' DGaoPC(data = data, m = 3, n1 = 20, K = 5)
#'
#' @export
DGaoPC <- function(data, m, n1, K) {
  SigmahatG1 <- list()
  SigmahatG2 <- list()
  AG1 <- list()
  AG2 <- list()
  DG1 <- list()
  DG2 <- list()
  DG3 <- list()
  F1hat <- list()
  sGhat <- list()
  
  for (i in 1:K) {
    n <- nrow(data)
    pc <- 2
    L <- matrix(rep(0, K * n1), ncol = n1)
    R <- matrix(0, n1, n, ncol = n)
    L[i, ] <- sample(1:n, n1, replace = FALSE)
    r <- matrix(c(1:n1, L[i, ]), ncol = n1, byrow = TRUE)
    R[t(r)] <- 1
    X1 <- R %*% as.matrix(data)
    X <- scale(X1)
    
    SigmahatG1[[i]] <- cor(X)
    eig3 <- eigen(SigmahatG1[[i]])
    lambda1hat <- eig3$values[1:m]
    ind <- order(lambda1hat, decreasing = TRUE)
    Q <- eig3$vectors[, ind]
    AG1[[i]] <- Q[, 1:m]
    hG1 <- diag(AG1[[i]] %*% t(AG1[[i]]))
    DG1[[i]] <- diag(SigmahatG1[[i]] - hG1)
    
    F1hat[[i]] <- X %*% AG1[[i]]
    F1star <- F1hat[[i]] / sqrt(n)
    SigmahatG2[[i]] <- cor(F1star)
    eig4 <- eigen(SigmahatG2[[i]])
    lambda2hat <- eig4$values[1:pc]
    ind <- order(lambda2hat, decreasing = TRUE)
    Q <- eig4$vectors[, ind]
    AG2[[i]] <- Q[, 1:pc]
    hG2 <- diag(AG2[[i]] %*% t(AG2[[i]]))
    DG2[[i]] <- diag(SigmahatG2[[i]] - hG2)
    
    Fhat <- F1star %*% AG2[[i]]
    XGhat <- Fhat %*% t(AG2[[i]]) %*% t(AG1[[i]])
    sGhat[[i]] <- cov(XGhat)
    hG3 <- diag(t(t(AG2[[i]]) %*% t(AG1[[i]])) %*% (t(AG2[[i]]) %*% t(AG1[[i]])))
    DG3[[i]] <- diag(sGhat[[i]] - hG3)
  }
  
  return(list(AG1 = AG1, AG2 = AG2, DG3 = DG3, sGhat = sGhat))
}