# ---------- plots ----------
#' Preliminary diagnostic plots
#'
#' @param x numeric vector
#' @param which character vector: "hist", "qq", "pp", "logqq"
#' @param hist_bins number of bins for histogram
#' @param col_hist color for histogram
#' @param col_density color for density line in histogram
#' @param col_qq color for qq points
#' @param col_line color for lines in "qq", "pp", "logqq" plots
#' @return no return value, called for side effects (generating plots)
#' @export
prelim_plots <- function(x, which = c("hist"), hist_bins = 60,
                         col_hist = "grey85", col_density = "darkorange",
                         col_qq = "grey60", col_line = "darkorange") {
  which <- match.arg(which, choices = c("hist", "qq", "pp", "logqq"), several.ok = TRUE)
  np <- length(which); op <- par(no.readonly = TRUE); on.exit(par(op))
  nrow <- if (np <= 2) 1 else ceiling(np / 2); par(mfrow = c(nrow, min(2, np)))
  if ("hist" %in% which) {
    hist(x, breaks = hist_bins, col = col_hist, main = "Histogram with density",
         xlab = "x", freq = FALSE)
    lines(density(x, na.rm = TRUE), col = col_density, lwd = 2)
  }
  if ("qq" %in% which) {
    qqnorm(x, main = "QQ-plot (Normal)", col = col_qq)
    qqline(x, col = col_line, lwd = 2)
  }
  if ("pp" %in% which) {
    ecdfv <- ecdf(x); pvals <- ecdfv(x)
    plot(sort(pvals), type = "l", main = "PP-plot",
         xlab = "Obs index", ylab = "ECDF", col = col_line, lwd = 2)
    abline(0, 1 / length(x), col = "black", lty = "dashed")
  }
  if ("logqq" %in% which) {
    if (any(x <= 0)) stop("logqq requires positive x")
    lx <- log(x)
    qqnorm(lx, main = "QQ-plot (log x)", col = col_qq)
    qqline(lx, col = col_line, lwd = 2)
  }
}
