% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/connectivity_estimation.distributions.R
\encoding{UTF-8}
\name{optim.rel.conn.dists}
\alias{optim.rel.conn.dists}
\title{Maximum-likelihood estimate for relative connectivity given two distributions
for scores for marked and unmarked individuals}
\usage{
optim.rel.conn.dists(
  obs,
  d.unmarked,
  d.marked,
  p = 1,
  phi0 = 0.5,
  method = "Brent",
  lower = 0,
  upper = 1,
  ...
)
}
\arguments{
\item{obs}{Vector of observed score values for potentially marked individuals}

\item{d.unmarked}{A function representing the PDF of unmarked individuals. 
Must be normalized so that it integrates to 1 for the function to work 
properly.}

\item{d.marked}{A function representing the PDF of marked individuals.  Must 
be normalized so that it integrates to 1 for the function to work properly.}

\item{p}{Fraction of individuals (i.e., eggs) marked in the source population}

\item{phi0}{Initial value for \eqn{\phi}, the fraction of settlers at the 
destination population that originated at the source population, for 
\code{\link{optim}} function. Defaults to 0.5.}

\item{method}{Method variable for \code{\link{optim}} function. Defaults to 
\code{"Brent"}.}

\item{lower}{Lower limit for search for fraction of marked individuals. 
Defaults to 0.}

\item{upper}{Upper limit for search for fraction of marked individuals. 
Defaults to 1.}

\item{\dots}{Additional arguments for the \code{\link{optim}} function.}
}
\value{
A list with results of optimization. Optimal fraction of marked 
  individuals is in \code{phi} field. Negative log-likelihood is in the 
  \code{neg.log.prob} field. See \code{\link{optim}} for other elements of
  list.
}
\description{
This function calculates the value for relative connectivity that best fits a
set of observed score values, a pair of distributions for marked and unmarked
individuals and an estimate of the fraction of eggs marked in the source 
population, \code{p}.
}
\examples{
library(ConnMatTools)
data(damselfish.lods)

# Histograms of simulated LODs
l <- seq(-1,30,0.5)
h.in <- hist(damselfish.lods$in.group,breaks=l)
h.out <- hist(damselfish.lods$out.group,breaks=l)

# PDFs for marked and unmarked individuals based on simulations
d.marked <- stepfun.hist(h.in)
d.unmarked <- stepfun.hist(h.out)

# Fraction of adults genotyped at source site
p.adults <- 0.25

# prior.shape1=1 # Uniform prior
prior.shape1=0.5 # Jeffreys prior

# Fraction of eggs from one or more genotyped parents
p <- dual.mark.transmission(p.adults)$p

# PDF for relative connectivity
D <- d.rel.conn.dists.func(damselfish.lods$real.children,
                           d.unmarked,d.marked,p,
                           prior.shape1=prior.shape1)

# Estimate most probable value for relative connectivity
phi.mx <- optim.rel.conn.dists(damselfish.lods$real.children,
                                    d.unmarked,d.marked,p)$phi

# Estimate 95\% confidence interval for relative connectivity
Q <- q.rel.conn.dists.func(damselfish.lods$real.children,
                           d.unmarked,d.marked,p,
                           prior.shape1=prior.shape1)

# Plot it up
phi <- seq(0,1,0.001)
plot(phi,D(phi),type="l",
     xlim=c(0,0.1),
     main="PDF for relative connectivity",
     xlab=expression(phi),
     ylab="Probability density")

abline(v=phi.mx,col="green",lty="dashed")
abline(v=Q(c(0.025,0.975)),col="red",lty="dashed")
}
\references{
Kaplan DM, Cuif M, Fauvelot C, Vigliola L, Nguyen-Huu T, Tiavouane J and Lett C 
  (in press) Uncertainty in empirical estimates of marine larval connectivity. 
  ICES Journal of Marine Science. doi:10.1093/icesjms/fsw182.
}
\seealso{
Other connectivity estimation: 
\code{\link{d.rel.conn.beta.prior}()},
\code{\link{d.rel.conn.dists.func}()},
\code{\link{d.rel.conn.finite.settlement}()},
\code{\link{d.rel.conn.multinomial.unnorm}()},
\code{\link{d.rel.conn.multiple}()},
\code{\link{d.rel.conn.unif.prior}()},
\code{\link{dual.mark.transmission}()},
\code{\link{r.marked.egg.fraction}()}
}
\author{
David M. Kaplan \email{dmkaplan2000@gmail.com}
}
\concept{connectivity estimation}
