% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simCAT.R
\name{simCAT}
\alias{simCAT}
\title{CAT simulation}
\usage{
simCAT(
  resps,
  bank,
  model = "3PL",
  start.theta = 0,
  sel.method = "MFI",
  cat.type = "variable",
  acceleration = 1,
  met.weight = "mcclarty",
  threshold = 0.3,
  rmax = 1,
  content.names = NULL,
  content.props = NULL,
  content.items = NULL,
  met.content = "MCCAT",
  stop = list(se = 0.3, hypo = 0.015, hyper = Inf),
  progress = TRUE
)
}
\arguments{
\item{resps}{a matrix with responses (0 and 1).
The number of columns
corresponds to the number of items}

\item{bank}{matrix with item parameters (a, b, c)}

\item{model}{may be \verb{3PL} or \code{graded}}

\item{start.theta}{first theta}

\item{sel.method}{item selection method: may be \code{MFI}, \code{progressive}
or \code{random}}

\item{cat.type}{CAT with \code{variable} or \code{fixed} length
Necessary only for progressive method.}

\item{acceleration}{acceleration parameter.
Necessary only for progressive method.}

\item{met.weight}{the procedure to calculate the \code{progressive}'s weight in variable-length
CAT. It can be \code{"magis"} or \code{"mcclarty"} (default). See details.}

\item{threshold}{threshold for \code{cat.type}.
Necessary only for progressive method.}

\item{rmax}{item maximum exposure rate}

\item{content.names}{vector with the contents of the test}

\item{content.props}{desirable proportion of each content in test, in
the same order of \code{content.names}}

\item{content.items}{vector indicating the content of each item}

\item{met.content}{content balancing method: \code{MCCAT} (default), \code{CCAT}
#' or \code{MMM}. See \code{content.balancing} for more information.}

\item{stop}{list with stopping rule and thresholds
\itemize{
\item \code{se} minimum standard error
\item \code{delta.theta} minimum absolute difference between current and previous theta
\item \code{hypo} minimum standard error reduction
\item \code{hyper} minimum standard error reduction after achieving \code{se}
\item \code{info} maximum information of an available item
\item \code{max.items} maximum number of items
\item \code{min.items} maximum number of items
\item \code{fixed} fixed number of items
}}

\item{progress}{shows progress bar}
}
\value{
a list with five elements
\itemize{
\item \code{score} estimated theta
\item \code{convergence} \code{TRUE} if the application ended before reaching the maximum test length
\item \code{theta.history} estimated theta after each item administration
\item \code{se.history} standard error after each item administration
\item \code{prev.resps} previous responses (administered items)
}
}
\description{
A CAT simulation with dichotomous items.
}
\details{
For details about formula of selection methods, see \code{\link[simCAT]{select.item}}.
}
\examples{

\donttest{
set.seed(1)
n.items <- 50
pars <- data.frame(
 a = rlnorm(n.items),
 b = rnorm(n.items),
 c = rbeta(n.items, 5, 17),
 d = 1)

# thetas
theta <- rnorm(100)

# simulate responses
resps <- gen.resp(theta, pars[,1:3])

results <- simCAT(resps = resps,
 bank = pars[,1:3],
 start.theta = 0,
 sel.method = 'MFI',
 cat.type = 'variable',
 threshold = .3,
 stop = list(se = .3, max.items = 10))

eval <- cat.evaluation(
 results = results,
 true.scores = theta,
 item.name = paste0('I', 1:nrow(pars)),
 rmax = 1)

#### 3 replications
replications <- 3

# simulate responses
set.seed(1)
resps <- list()
for(i in 1:replications)
 resps[[i]] <- gen.resp(theta, pars[,1:3])

# CAT
results <- list()
for (rep in 1:replications)
{
 print(paste0('replication: ', rep, '/', replications))
 results[[rep]] <- simCAT(
  resps = resps[[rep]],
  bank = pars[,1:3],
  start.theta = 0,
  sel.method = 'MFI',
  cat.type = 'variable',
  threshold = .3,
  stop = list(se = .5, max.items = 10))
}

eval <- cat.evaluation(
 results = results,
 true.scores = theta,
 item.name = paste0('I', 1:nrow(pars)),
 rmax = 1)
}

}
\references{
Barrada, J. R., Olea, J., Ponsoda, V., & Abad, F. J. (2008). \emph{Incorporating randomness in the Fisher information for improving item-exposure control in CATs}. British Journal of Mathematical and Statistical Psychology, 61(2), 493–513. 10.1348/000711007X230937

Leroux, A. J., & Dodd, B. G. (2016). \emph{A comparison of exposure control procedures in CATs using the GPC model}. The Journal of Experimental Education, 84(4), 666–685. 10.1080/00220973.2015.1099511

Magis, D., & Barrada, J. R. (2017). \emph{Computerized adaptive testing with R: recent updates of the package catR}. Journal of Statistical Software, 76(Code Snippet 1). 10.18637/jss.v076.c01

McClarty, K. L., Sperling, R. A., & Dodd, B. G. (2006). \emph{A variant of the progressive-restricted item exposure control procedure in computerized adaptive testing}. Annual Meeting of the American Educational Research Association, San Francisco
}
\author{
Alexandre Jaloto
}
