#' @title Summary of portfolio backtest
#' 
#' @description Summarize the results from a portfolio backtest.
#' 
#' @param bt Backtest results as produced by the function \code{\link{portfolioBacktest}}.
#' @param portfolio_indexes Numerical vector of portfolio indexes whose performance will be summarized, 
#'                          e.g., \code{c(1, 2)} means to summarize the performance of the first and 
#'                          second portfolios recorded in \code{bt}.
#' @param portfolio_names String vector of portfolio names whose performance will be summarized, 
#'                        e.g., \code{c("Uniform", "GMVP")} means to summarize the performance of portfolios
#'                        with names \code{"Uniform"} and \code{"GMVP"} in \code{bt} (default is 
#'                        \code{names(bt)} except the benchmark names). Only considered when 
#'                        \code{portfolio_indexes} is not passed.
#' @param summary_fun Summary function to be employed (e.g., \code{median} or \code{mean}).
#' @param show_benchmark Logical value indicating whether to include benchmarks in the summary (default is \code{TRUE}).
#' 
#' @return List with the following elements:
#' \item{\code{performance_summary}}{Performance criteria: 
#'                 \code{"Sharpe ratio"}, \code{"max drawdown"}, \code{"annual return"}, \code{"annual volatility"}, 
#'                 \code{"Sterling ratio"}, \code{"Omega ratio"}, and \code{"ROT bps"}.
#'                  Default is \code{"Sharpe ratio"}.}
#' \item{\code{failure_rate}}{Failure rate of each portfolio (failure is when the portfolio
#'                            function generates an error or the maximum CPU time is exceeded).}
#' \item{\code{cpu_time_summary}}{Summary of the CPU usage by each portfolio function.}
#' \item{\code{error_message}}{Error messages generated by each portfolio function over each dataset.
#'                             Useful for debugging purposes.}
#' 
#' @author Rui Zhou and Daniel P. Palomar
#' 
#' @examples
#' library(portfolioBacktest)
#' data(dataset10)  # load dataset
#' 
#' # define your own portfolio function
#' uniform_portfolio <- function(dataset) {
#'   N <- ncol(dataset$adjusted)
#'   return(rep(1/N, N))
#' }
#' 
#' # do backtest
#' bt <- portfolioBacktest(list("Uniform" = uniform_portfolio), dataset10)
#' 
#' # show the summary
#' bt_sum <- backtestSummary(bt)
#' names(bt_sum)
#' bt_sum$performance_summary
#' 
#' @export
backtestSummary <- function(bt, portfolio_indexes = NA, portfolio_names = NA, 
                            summary_fun = median, show_benchmark = TRUE) {
  if (anyNA(portfolio_names) && anyNA(portfolio_indexes)) portfolio_indexes <- attr(bt, 'portfolio_index')
  if (!anyNA(portfolio_indexes)) portfolio_names <- names(bt)[portfolio_indexes]
  if (show_benchmark) portfolio_names <- c(portfolio_names, names(bt)[attr(bt, 'benchmark_index')])
  
  n_portfolio <- length(portfolio_names)
  result <- list()
  summary_container <- matrix(NA, n_portfolio, length(portfolioPerformance()))
  
  performance <- failure_rate <- cpu_time_summary <- list()
  res_table <- backtestTable(bt)
  for (portfolio_name in portfolio_names) {
    tmp <- backtestSummarySinglePortfolio(res_table, portfolio_name, summary_fun)
    performance[[portfolio_name]] <- tmp$performance
    failure_rate[[portfolio_name]] <- tmp$failure_rate
    cpu_time_summary[[portfolio_name]] <- tmp$cpu_time_summary
  }
  
  rt <- list()
  rt$performance_summary <- cbind(Reduce(cbind, performance))
  rt$failure_rate <- unlist(failure_rate)
  rt$cpu_time_summary <- unlist(cpu_time_summary)
  rt$error_message <- res_table$error_message
  
  colnames(rt$performance_summary) <- names(rt$failure_rate) <- names(rt$cpu_time_summary) <- portfolio_names
  rownames(rt$performance_summary) <- names(portfolioPerformance())
  return(rt)
}


backtestSummarySinglePortfolio <- function(res_table, portfolio_name, summary_fun) {
  # assume the res_table contains all performance metric
  performance <- portfolioPerformance()
  mask_performance <- names(performance)
  fail_mask <- res_table$error[, portfolio_name]
  failure_rate <- mean(fail_mask)
  cpu_time_summary <- NA
  if (failure_rate < 1) {
    for (metric in mask_performance)
      performance[metric] <- summary_fun(res_table[[metric]][!fail_mask, portfolio_name])
    cpu_time_summary <- summary_fun(res_table$cpu_time[!fail_mask, portfolio_name])
  }
  return(list(performance = performance, 
              failure_rate = failure_rate,
              cpu_time_summary = cpu_time_summary))
}




#' @title Table with portfolio backtest results
#' 
#' @description Create table with the results from a portfolio backtest.
#' 
#' @inheritParams backtestSummary
#' @param measures String vector to select performane measures (default is all) from
#'                 \code{"Sharpe ratio"}, \code{"max drawdown"}, \code{"annual return"}, \code{"annual volatility"}, 
#'                 \code{"Sterling ratio"}, \code{"Omega ratio"}, \code{"ROT bps"},
#'                 \code{"error"}, \code{"cpu_time"}, and \code{"error_message"}.
#' 
#' @return List with the following elements:
#' \item{\code{<performance criterion>}}{One item per performance measures as selected by argument \code{measures}.}
#' \item{\code{error}}{Error status (\code{TRUE} or \code{FALSE}) for each portfolio over each dataset
#'                    (\code{TRUE} is when the portfolio function generates an error or the maximum CPU time is exceeded).}
#' \item{\code{cpu_time}}{CPU usage by each portfolio function over each dataset.}
#' \item{\code{error_message}}{Error messages generated by each portfolio function over each dataset.
#'                             Useful for debugging purposes.}
#' 
#' @author Rui Zhou and Daniel P. Palomar
#' 
#' @examples 
#' library(portfolioBacktest)
#' data(dataset10)  # load dataset
#' 
#' # define your own portfolio function
#' uniform_portfolio <- function(dataset) {
#'   N <- ncol(dataset$adjusted)
#'   return(rep(1/N, N))
#' }
#' 
#' # do backtest
#' bt <- portfolioBacktest(list("Uniform" = uniform_portfolio), dataset10)
#' 
#' # show the backtest results in table
#' bt_tab <- backtestTable(bt)
#' bt_tab[c("Sharpe ratio", "max drawdown")]
#' 
#' @export  
backtestTable <- function(bt, portfolio_indexes = NA, portfolio_names = NA, 
                          show_benchmark = TRUE, measures = NULL) {
  # check portfolio index and names
  if (anyNA(portfolio_names) && anyNA(portfolio_indexes)) portfolio_indexes <- attr(bt, 'portfolio_index')
  if (!anyNA(portfolio_indexes)) portfolio_names <- names(bt)[portfolio_indexes]
  if (show_benchmark) portfolio_names <- c(portfolio_names, names(bt)[attr(bt, 'benchmark_index')])
  
  # check measures
  measures_range <- c(names(portfolioPerformance()), 'error', 'error_message', 'cpu_time')
  if (is.null(measures)) measures <- measures_range
  if (any(!(measures %in% measures_range))) stop("\"measures\" contains invalid element.")
  
  # check if source_error happen
  valid_mask <- sapply(bt[portfolio_names], function(x){is.null(x$source_error_message)})
  if (!any(valid_mask)) stop("all files fail to be sourced")
  
  # extract results and combine into matrix
  N_dataset <- length(bt[[portfolio_names[valid_mask][1]]])
  N_portfolio <- length(portfolio_names)
  mask_performance <- setdiff(measures, c('error', 'error_message', 'cpu_time'))
  
  
  container <- matrix(NA, N_dataset, N_portfolio)
  colnames(container) <- portfolio_names
  rownames(container) <- names(bt[[1]])
  cpu_time <- error <- container
  performance <- error_message <- list()
  
  # fill in all results
  for (i in 1:N_portfolio) {
    
    tmp <- backtestSelector(bt = bt, portfolio_name = portfolio_names[i], measures = measures)
    
    for (metric in mask_performance) {
      # creat space in first visit
      if (i == 1) performance[[metric]] <- container
      # check source error
      if (!valid_mask[i]) next
      # fill in certain metric
      performance[[metric]][, i] <- tmp$performance[, metric]
    }
    
    if ('error' %in% measures)
      if (valid_mask[i]) 
        error[, i] <- tmp$error
      else
        error[, i] <- TRUE
      
    if ('cpu_time' %in% measures)
      if (valid_mask[i])
        cpu_time[, i] <- tmp$cpu_time
    
    if ('error_message' %in% measures)
      if (valid_mask[i])
        error_message[[portfolio_names[i]]] <- tmp$error_message
      else
        error_message[[portfolio_names[i]]] <- bt[[portfolio_names[i]]]$source_error_message
  }
  
  rt <- list()
  if (length(mask_performance) >= 1) rt <- performance
  if ('error' %in% measures)         rt$error <- error
  if ('cpu_time' %in% measures)      rt$cpu_time <- cpu_time
  if ('error_message' %in% measures) rt$error_message <- error_message
  
  return(rt)
}





#' @title Selector of portfolio backtest results
#' 
#' @description Select the results from a portfolio backtest.
#' 
#' @inheritParams backtestSummary
#' @param portfolio_index Index number of a portfolio, e.g., \code{1} means to select the performance 
#'                        of the first portfolio recorded in \code{bt}.
#' @param portfolio_name String name of a portfolio, e.g., \code{"GMVP"} means to select the performance 
#'                       of portfolio with name \code{"GMVP"} in \code{bt}. 
#'                       Only considered when \code{portfolio_index} is not passed.
#' @param measures String vector to select performane measures (default is all) from
#'                 \code{"Sharpe ratio"}, \code{"max drawdown"}, \code{"annual return"}, \code{"annual volatility"}, 
#'                 \code{"Sterling ratio"}, \code{"Omega ratio"}, and \code{"ROT bps"}.
#' 
#' @return List with the following elements:
#' \item{\code{performance}}{Performance measures selected by argument \code{measures}.}
#' \item{\code{error}}{Error status (\code{TRUE} or \code{FALSE}) of portfolio over each dataset
#'                    (\code{TRUE} is when the portfolio function generates an error or the maximum CPU time is exceeded).}
#' \item{\code{error_message}}{Error messages generated by portfolio function over each dataset.
#'                             Useful for debugging purposes.}
#' \item{\code{cpu_time}}{CPU usage by portfolio function over each dataset.}
#' \item{\code{portfolio}}{Portfolio weights generated by portfolio function over each dataset.}
#' \item{\code{return}}{Portfolio returns over each dataset.}
#' \item{\code{wealth}}{Portfolio wealth (aka cumulative returns or cumulative P&L) over each dataset.}
#' 
#' @author Rui Zhou and Daniel P. Palomar
#' 
#' @examples 
#' library(portfolioBacktest)
#' data("dataset10")  # load dataset
#' 
#' # define your own portfolio function
#' uniform_portfolio <- function(dataset) {
#'   N <- ncol(dataset$adjusted)
#'   return(rep(1/N, N))
#' }
#' 
#' # do backtest
#' bt <- portfolioBacktest(list("Uniform" = uniform_portfolio), dataset10)
#' 
#' # extract your interested portfolio result
#' bt_sel <- backtestSelector(bt, portfolio_name = "Uniform")
#' names(bt_sel)
#' 
#' @export
#' 
backtestSelector <- function(bt, portfolio_index = NULL, portfolio_name = NULL, measures = NULL) {
  measures_range <- c(names(portfolioPerformance()), 'error', 'error_message', 'cpu_time', 'return', 'portfolio')
  if (is.null(measures)) measures <- measures_range
  if (any(!(measures %in% measures_range))) stop("\"measures\" contains invalid element.")
  if (length(measures) == 0) stop("\"measures\" must have length > 1.")
  if (length(portfolio_name) > 1 || length(portfolio_index) > 1) stop("Only one portfolio can be selected.")
  if (is.null(portfolio_name) && is.null(portfolio_index)) stop("must select a portfolio.") 
  if (!is.null(portfolio_index)) portfolio_name <- names(bt)[portfolio_index]
  if (!is.null(bt[[portfolio_name]]$source_error_message)) return(bt[[portfolio_name]])
  
  result <- list()
  mask_performance <- setdiff(measures, c('error', 'error_message', 'cpu_time', 'return', 'portfolio'))
  if (length(mask_performance) > 0)
    result$performance <- t(sapply(bt[[portfolio_name]], function(x){x$performance[mask_performance]}))
  if ('error' %in% measures) 
    result$error <- sapply(bt[[portfolio_name]], function(x){x$error})
  if ('error_message' %in% measures) 
    result$error_message <- lapply(bt[[portfolio_name]], function(x){x$error_message})
  if ('cpu_time' %in% measures)
    result$cpu_time <- sapply(bt[[portfolio_name]], function(x){x$cpu_time})
  if ('portfolio' %in% measures)
    result$portfolio <- lapply(bt[[portfolio_name]], function(x){x$portfolio})
  if ('return' %in% measures) {
    result$return <- lapply(bt[[portfolio_name]], function(x){x$return})
    result$wealth <- lapply(bt[[portfolio_name]], function(x){x$wealth})
  }
  
  return(result)
}
