% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{binary_tte_sim}
\alias{binary_tte_sim}
\title{Simulation for a binary endpoint and a time-to-event endpoint}
\usage{
binary_tte_sim(
  kMax1 = 1L,
  kMax2 = 1L,
  riskDiffH0 = 0,
  hazardRatioH0 = 1,
  allocation1 = 1L,
  allocation2 = 1L,
  accrualTime = 0L,
  accrualIntensity = NA_real_,
  piecewiseSurvivalTime = 0L,
  stratumFraction = 1L,
  globalOddsRatio = 1,
  pi1 = NA_real_,
  pi2 = NA_real_,
  lambda1 = NA_real_,
  lambda2 = NA_real_,
  gamma1 = 0L,
  gamma2 = 0L,
  delta1 = 0L,
  delta2 = 0L,
  upper1 = NA_real_,
  upper2 = NA_real_,
  accrualDuration = NA_real_,
  plannedTime = NA_real_,
  plannedEvents = NA_integer_,
  maxNumberOfIterations = 1000L,
  maxNumberOfRawDatasetsPerStage = 0L,
  seed = NA_integer_
)
}
\arguments{
\item{kMax1}{Number of stages for the binary endpoint.}

\item{kMax2}{Number of stages for the time-to-event endpoint.}

\item{riskDiffH0}{Risk difference under the null hypothesis for the
binary endpoint.}

\item{hazardRatioH0}{Hazard ratio under the null hypothesis for the
time-to-event endpoint.}

\item{allocation1}{Number of subjects in the treatment group in
a randomization block. Defaults to 1 for equal randomization.}

\item{allocation2}{Number of subjects in the control group in
a randomization block. Defaults to 1 for equal randomization.}

\item{accrualTime}{A vector that specifies the starting time of
piecewise Poisson enrollment time intervals. Must start with 0, e.g.,
\code{c(0, 3)} breaks the time axis into 2 accrual intervals:
[0, 3) and [3, Inf).}

\item{accrualIntensity}{A vector of accrual intensities. One for
each accrual time interval.}

\item{piecewiseSurvivalTime}{A vector that specifies the starting time of
piecewise exponential survival time intervals. Must start with 0, e.g.,
\code{c(0, 6)} breaks the time axis into 2 event intervals:
[0, 6) and [6, Inf).
Defaults to 0 for exponential distribution.}

\item{stratumFraction}{A vector of stratum fractions that sum to 1.
Defaults to 1 for no stratification.}

\item{globalOddsRatio}{The global odds ratio of the Plackett copula.}

\item{pi1}{Response probabilities by stratum for the treatment group
for the binary endpoint.}

\item{pi2}{Response probabilities by stratum for the control group
for the binary endpoint.}

\item{lambda1}{A vector of hazard rates for the event in each analysis
time interval by stratum for the treatment group for the time-to-event
endpoint.}

\item{lambda2}{A vector of hazard rates for the event in each analysis
time interval by stratum for the control group for the time-to-event
endpoint.}

\item{gamma1}{The hazard rate for exponential dropout, a vector of
hazard rates for piecewise exponential dropout applicable for all
strata, or a vector of hazard rates for dropout in each analysis time
interval by stratum for the treatment group.}

\item{gamma2}{The hazard rate for exponential dropout, a vector of
hazard rates for piecewise exponential dropout applicable for all
strata, or a vector of hazard rates for dropout in each analysis time
interval by stratum for the control group.}

\item{delta1}{The hazard rate for exponential treatment discontinuation,
a vector of hazard rates for piecewise exponential treatment
discontinuation applicable for all strata, or a vector of hazard rates
for treatment discontinuation in each analysis time interval by
stratum for the treatment group for the binary endpoint.}

\item{delta2}{The hazard rate for exponential treatment discontinuation,
a vector of hazard rates for piecewise exponential treatment
discontinuation applicable for all strata, or a vector of hazard rates
for treatment discontinuation in each analysis time interval by
stratum for the control group for the binary endpoint.}

\item{upper1}{The protocol-specified treatment duration for the treatment
group.}

\item{upper2}{The protocol-specified treatment duration for the control
group.}

\item{accrualDuration}{Duration of the enrollment period.}

\item{plannedTime}{The calendar times for the analyses of the binary
endpoint.}

\item{plannedEvents}{The planned cumulative total number of events for
the time-to-event endpoint.}

\item{maxNumberOfIterations}{The number of simulation iterations.}

\item{maxNumberOfRawDatasetsPerStage}{The number of raw datasets per
stage to extract.}

\item{seed}{The seed to reproduce the simulation results.
The seed from the environment will be used if left unspecified,}
}
\value{
A list with 4 components:
\itemize{
\item \code{sumdataBIN}: A data frame of summary data by iteration and stage
for the binary endpoint:
\itemize{
\item \code{iterationNumber}: The iteration number.
\item \code{stageNumber}: The stage number, covering all stages even if
the trial stops at an interim look.
\item \code{analysisTime}: The time for the stage since trial start.
\item \code{accruals1}: The number of subjects enrolled at the stage for
the treatment group.
\item \code{accruals2}: The number of subjects enrolled at the stage for
the control group.
\item \code{totalAccruals}: The total number of subjects enrolled at
the stage.
\item \code{source1}: The total number of subjects with response status
determined by the underlying latent response variable.
\item \code{source2}: The total number of subjects with response status
(non-responder) determined by experiencing the event for the
time-to-event endpoint.
\item \code{source3}: The total number of subjects with response status
(non-responder) determined by dropping out prior to the PTFU1
visit.
\item \code{n1}: The number of subjects included in the analysis of
the binary endpoint for the treatment group.
\item \code{n2}: The number of subjects included in the analysis of
the binary endpoint for the control group.
\item \code{n}: The total number of subjects included in the analysis of
the binary endpoint at the stage.
\item \code{y1}: The number of responders for the binary endpoint in
the treatment group.
\item \code{y2}: The number of responders for the binary endpoint in
the control group.
\item \code{y}: The total number of responders for the binary endpoint
at the stage.
\item \code{riskDiff}: The estimated risk difference for the binary
endpoint.
\item \code{seRiskDiff}: The standard error for risk difference based on
the Sato approximation.
\item \code{mnStatistic}: The Mantel-Haenszel test Z-statistic for
the binary endpoint.
}
\item \code{sumdataTTE}: A data frame of summary data by iteration and stage
for the time-to-event endpoint:
\itemize{
\item \code{iterationNumber}: The iteration number.
\item \code{eventsNotAchieved}: Whether the target number of events
is not achieved for the iteration.
\item \code{stageNumber}: The stage number, covering all stages even if
the trial stops at an interim look.
\item \code{analysisTime}: The time for the stage since trial start.
\item \code{accruals1}: The number of subjects enrolled at the stage for
the treatment group.
\item \code{accruals2}: The number of subjects enrolled at the stage for
the control group.
\item \code{totalAccruals}: The total number of subjects enrolled at
the stage.
\item \code{events1}: The number of events at the stage for
the treatment group.
\item \code{events2}: The number of events at the stage for
the control group.
\item \code{totalEvents}: The total number of events at the stage.
\item \code{dropouts1}: The number of dropouts at the stage for
the treatment group.
\item \code{dropouts2}: The number of dropouts at the stage for
the control group.
\item \code{totalDropouts}: The total number of dropouts at the stage.
\item \code{logRankStatistic}: The log-rank test Z-statistic for
the time-to-event endpoint.
}
\item \code{rawdataBIN} (exists if \code{maxNumberOfRawDatasetsPerStage} is a
positive integer): A data frame for subject-level data for the binary
endpoint for selected replications, containing the following variables:
\itemize{
\item \code{iterationNumber}: The iteration number.
\item \code{stageNumber}: The stage under consideration.
\item \code{analysisTime}: The time for the stage since trial start.
\item \code{subjectId}: The subject ID.
\item \code{arrivalTime}: The enrollment time for the subject.
\item \code{stratum}: The stratum for the subject.
\item \code{treatmentGroup}: The treatment group (1 or 2) for the
subject.
\item \code{survivalTime}: The underlying survival time for the
time-to-event endpoint for the subject.
\item \code{dropoutTime}: The underlying dropout time for the
time-to-event endpoint for the subject.
\item \code{ptfu1Time}:The underlying assessment time for the
binary endpoint for the subject.
\item \code{timeUnderObservation}: The time under observation
since randomization for the binary endpoint for the subject.
\item \code{responder}: Whether the subject is a responder for the
binary endpoint.
\item \code{source}: The source of the determination of responder
status for the binary endpoint: = 1 based on the underlying
latent response variable, = 2 based on the occurrence of
the time-to-event endpoint before the assessment time of the
binary endpoint (imputed as a non-responder), = 3 based on
the dropout before the assessment time of the binary endpoint
(imputed as a non-responder), = 4 excluded from analysis
due to administrative censoring.
}
\item \code{rawdataTTE} (exists if \code{maxNumberOfRawDatasetsPerStage} is a
positive integer): A data frame for subject-level data for the
time-to-event endpoint for selected replications, containing the
following variables:
\itemize{
\item \code{iterationNumber}: The iteration number.
\item \code{stageNumber}: The stage under consideration.
\item \code{analysisTime}: The time for the stage since trial start.
\item \code{subjectId}: The subject ID.
\item \code{arrivalTime}: The enrollment time for the subject.
\item \code{stratum}: The stratum for the subject.
\item \code{treatmentGroup}: The treatment group (1 or 2) for the
subject.
\item \code{survivalTime}: The underlying survival time for the
time-to-event endpoint for the subject.
\item \code{dropoutTime}: The underlying dropout time for the
time-to-event endpoint for the subject.
\item \code{timeUnderObservation}: The time under observation
since randomization for the time-to-event endpoint for the subject.
\item \code{event}: Whether the subject experienced the event for the
time-to-event endpoint.
\item \code{dropoutEvent}: Whether the subject dropped out for the
time-to-event endpoint.
}
}
}
\description{
Performs simulation for two-endpoint two-arm group
sequential trials. The first endpoint is a binary endpoint and
the Mantel-Haenszel test is used to test risk difference.
The second endpoint is a time-to-event endpoint and the log-rank
test is used to test the treatment difference. The analysis times
of the first endpoint are determined by the specified calendar times,
while the analysis times for the second endpoint is based on the
planned number of events at each look. The binary endpoint is
assessed at the first post-treatment follow-up visit.
}
\details{
We consider dual primary endpoints with endpoint 1 being a
binary endpoint and endpoint 2 being a time-to-event endpoint.
The analyses of endpoint 1 will be based on calendar times, while
the analyses of endpoint 2 will be based on the number of events.
Therefor the analyses of the two endpoints are not at the same
time points. The correlation between the two endpoints is
characterized by the global odds ratio of the Plackett copula.
In addition, the time-to-event endpoint will render the binary
endpoint as a non-responder, and so does the dropout. In addition,
the treatment discontinuation will impact the number of available
subjects for analysis. The administrative censoring will exclude
subjects from the analysis of the binary endpoint.
}
\examples{

tcut = c(0, 12, 36, 48)
surv = c(1, 0.95, 0.82, 0.74)
lambda2 = (log(surv[1:3]) - log(surv[2:4]))/(tcut[2:4] - tcut[1:3])

sim1 = binary_tte_sim(
  kMax1 = 1,
  kMax2 = 2,
  accrualTime = 0:8,
  accrualIntensity = c(((1:8) - 0.5)/8, 1)*40,
  piecewiseSurvivalTime = c(0,12,36),
  globalOddsRatio = 1,
  pi1 = 0.80,
  pi2 = 0.65,
  lambda1 = 0.65*lambda2,
  lambda2 = lambda2,
  gamma1 = -log(1-0.04)/12,
  gamma2 = -log(1-0.04)/12,
  delta1 = -log(1-0.02)/12,
  delta2 = -log(1-0.02)/12,
  upper1 = 15*28/30.4,
  upper2 = 12*28/30.4,
  accrualDuration = 20,
  plannedTime = 20 + 15*28/30.4,
  plannedEvents = c(130, 173),
  maxNumberOfIterations = 1000,
  maxNumberOfRawDatasetsPerStage = 1,
  seed = 314159)


}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
