% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/discretize_jointly.R
\name{discretize.jointly}
\alias{discretize.jointly}
\title{Discretize Multivariate Continuous Data by a Cluster-Preserving Grid}
\usage{
discretize.jointly(data, k = c(2:10), cluster_label = NULL, min_level = 2)
}
\arguments{
\item{data}{a matrix containing two or more continuous variables.
Columns are variables, rows are observations.}

\item{k}{either the number or range of clusters to be found on \code{data}.
The default is 2 to 10 clusters. If a range is specified, an optimal k in
the range is chosen to maximize the average silhouette width.
If \code{cluster_label} is specified, \code{k} is ignored.}

\item{cluster_label}{a vector of user-specified cluster labels for each observation
in \code{data}. The user is free to choose any clustering.
If unspecified, k-means clustering is used by default.}

\item{min_level}{the minimum number of levels along each dimension}
}
\value{
A list that contains four items:
\item{\code{D}}{a matrix that contains the discretized version of the original \code{data}.
Discretized values are one(1)-based.}

\item{\code{grid}}{a list of vectors containing decision boundaries for each variable/dimension.}

\item{\code{clabels}}{a vector containing cluster labels for each observation in \code{data}.}

\item{\code{csimilarity}}{a similarity score between clusters from joint discretization
\code{D} and cluster labels \code{clabels}. The score is the adjusted Rand index.}
}
\description{
Discretize multivariate continuous data using a grid that captures the joint distribution via
preserving clusters in the original data
}
\details{
The function implements algorithms described in \insertCite{Jwang2020BCB}{GridOnClusters}.
}
\examples{
# using a specified k
x = rnorm(100)
y = sin(x)
z = cos(x)
data = cbind(x, y, z)
discretized_data = discretize.jointly(data, k=5)$D

# using a range of k
x = rnorm(1000)
y = log1p(abs(x))
z = tan(x)
data = cbind(x, y, z)
discretized_data = discretize.jointly(data, k=c(3:10))$D

# using an alternate clustering method to k-means
library(cluster)
x = rnorm(1000)
y = log1p(abs(x))
z = sin(x)
data = cbind(x, y, z)

# pre-cluster the data using partition around medoids (PAM)
cluster_label = pam(x=data, diss = FALSE, metric = "euclidean", k = 5)$clustering
discretized_data = discretize.jointly(data, cluster_label = cluster_label)$D

}
\references{
\insertAllCited{}
}
\seealso{
See \link[Ckmeans.1d.dp]{Ckmeans.1d.dp} for discretizing univariate continuous data.
}
\author{
Jiandong Wang, Sajal Kumar and Mingzhou Song
}
