#' @title data.for.distance
#' @description Formats the data generated by create.survey.results into
#' a form suitable for import into Distance
#' @param object an object of class Survey.Results generated by create.survey.results
#' @param file path and filename if the user would like the results saved to file. Should contain the .txt file extension.
#' @param round the amount of decimal places to round the distances to
#' @param sep the field separator string for writing to file
#' @importFrom utils write.table
#' @export
#' @author L Marshall
data.for.distance <- function(object, file = NULL, round = 2, sep = "\t"){
  # get data tables
  ddf.dat <- object@ddf.data@ddf.dat
  obs.table <- object@obs.table@obs.table
  sample.table <- object@sample.table@sample.table
  region.table <- object@region.table@region.table
  # check that user requested dht tables
  if(nrow(obs.table) == 0){
    stop("The survey results must contain an observation, sample and region table. Please ensure you set dht.tables = TRUE when creating the survey results.")
  }
  # get covariate names (assume they are after x-y coordinates)
  dat.names <- names(ddf.dat)
  start.index <- which(dat.names == "y") + 1
  end.index <- ncol(ddf.dat)
  if(end.index >= start.index){
    cov.names <- dat.names[start.index:end.index]  
  }else{
    cov.names = ""
  }
  # add in empty rows for transects without sightings
  missing.transects <- which(!sample.table$Sample.Label %in% ddf.dat$transect.ID)
  temp.miss <- data.frame(Sample.Label = missing.transects)
  temp.miss <- merge(temp.miss, sample.table, by = "Sample.Label")
  no.miss.transects <- length(missing.transects)
  dat.names <- names(ddf.dat)
  temp.mat <- matrix(data = NA, nrow = no.miss.transects, ncol = length(dat.names), dimnames = list(NULL, dat.names))
  temp.dat <- as.data.frame(temp.mat)
  temp.dat$transect.ID <- missing.transects
  temp.dat$Sample.Label <- temp.miss$Sample.Label
  temp.dat$Region.Label <- temp.miss$Region.Label
  # merge data tables
  temp <- merge(obs.table, ddf.dat, by = "object")
  temp <- rbind(temp, temp.dat)
  temp2 <- merge(sample.table, temp, by = c("Sample.Label", "Region.Label"))
  dist.data <- merge(region.table, temp2, by = c("Region.Label"))
  # order by transect ID
  index <- order(dist.data$Sample.Label)
  dist.data <- dist.data[index,]
  # Only keep certain columns (remove duplicates)
  col.index <- which(names(dist.data) %in% c("Region.Label", "Area", "Sample.Label", "Effort", "length", "distance", cov.names))
  dist.data <- dist.data[,col.index]
  dist.data$distance <- round(dist.data$distance, round)
  # Save if requested or return
  if(is.null(file)){
    return(dist.data)  
  }else{
    write.table(dist.data, file = file, na = "", sep = sep, eol = "\r\n", row.names = FALSE)
    invisible(dist.data)
  }
}