% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fays_generalized_replication.R
\name{as_fays_gen_rep_design}
\alias{as_fays_gen_rep_design}
\title{Convert a survey design object to a replication design
using Fay's generalized replication method}
\usage{
as_fays_gen_rep_design(
  design,
  variance_estimator = NULL,
  aux_var_names = NULL,
  max_replicates = Inf,
  balanced = TRUE,
  psd_option = "warn",
  mse = TRUE,
  compress = TRUE
)
}
\arguments{
\item{design}{A survey design object created using the 'survey' (or 'srvyr') package,
with class \code{'survey.design'} or \code{'svyimputationList'}.}

\item{variance_estimator}{The name of the variance estimator
whose quadratic form matrix should be created.
See \link[svrep]{variance-estimators} for a
detailed description of each variance estimator.
Options include:
\itemize{
  \item \strong{"Yates-Grundy"}: \cr The Yates-Grundy variance estimator based on
    first-order and second-order inclusion probabilities.
  \item \strong{"Horvitz-Thompson"}: \cr The Horvitz-Thompson variance estimator based on
    first-order and second-order inclusion probabilities.
  \item \strong{"Poisson Horvitz-Thompson"}: \cr The Horvitz-Thompson variance estimator
    based on assuming Poisson sampling, with first-order inclusion probabilities
    inferred from the sampling probabilities of the survey design object.
  \item \strong{"Stratified Multistage SRS"}: \cr The usual stratified multistage variance estimator
    based on estimating the variance of cluster totals within strata at each stage.
  \item \strong{"Ultimate Cluster"}: \cr The usual variance estimator based on estimating
    the variance of first-stage cluster totals within first-stage strata.
  \item \strong{"Deville-1"}: \cr A variance estimator for unequal-probability
    sampling without replacement, described in Matei and Tillé (2005)
    as "Deville 1".
  \item \strong{"Deville-2"}: \cr A variance estimator for unequal-probability
    sampling without replacement, described in Matei and Tillé (2005) as "Deville 2".
  \item \strong{"Deville-Tille": } \cr A variance estimator useful
    for balanced sampling designs, proposed by Deville and Tillé (2005).
  \item \strong{"SD1"}: \cr The non-circular successive-differences variance estimator described by Ash (2014),
    sometimes used for variance estimation for systematic sampling.
  \item \strong{"SD2"}:  \cr The circular successive-differences variance estimator described by Ash (2014).
    This estimator is the basis of the "successive-differences replication" estimator commonly used
    for variance estimation for systematic sampling.
  \item \strong{"BOSB"}: \cr The kernel-based variance estimator proposed by
    Breidt, Opsomer, and Sanchez-Borrego (2016) for use with systematic samples
    or other finely stratified designs. Uses the Epanechnikov kernel
    with the bandwidth automatically chosen to result in the smallest possible
    nonempty kernel window.
  \item\strong{"Beaumont-Emond"}: \cr The variance estimator of Beaumont and Emond (2022)
    for multistage unequal-probability sampling without replacement.
}}

\item{aux_var_names}{(Only used if \code{variance_estimator = "Deville-Tille")}.
A vector of the names of auxiliary variables used in sampling.}

\item{max_replicates}{The maximum number of replicates to allow (should be as large as possible, given computer memory/storage limitations).
A commonly-recommended default is 500. If the number of replicates needed
for a balanced, fully-efficient estimator is less than \code{max_replicates},
then only the number of replicates needed will be created.
If more replicates are needed than \code{max_replicates}, then the full number of replicates
needed will be created, but only a random subsample will be retained.}

\item{balanced}{If \code{balanced=TRUE}, the replicates
will all contribute equally to variance estimates, but
the number of replicates needed may slightly increase.}

\item{psd_option}{Either \code{"warn"} (the default) or \code{"error"}.
This option specifies what will happen if the target variance estimator
has a quadratic form matrix which is not positive semidefinite. This
can occasionally happen, particularly for two-phase designs. \cr
If \code{psd_option="error"}, then an error message will be displayed. \cr
If \code{psd_option="warn"}, then a warning message will be displayed,
and the quadratic form matrix will be approximated by the most similar
positive semidefinite matrix.
This approximation was suggested by Beaumont and Patak (2012),
who note that this is conservative in the sense of producing
overestimates of variance.
Beaumont and Patak (2012) argue that this overestimation is expected to be
small in magnitude. See \code{\link[svrep]{get_nearest_psd_matrix}}
for details of the approximation.}

\item{mse}{If \code{TRUE} (the default), compute variances from sums of squares around the point estimate from the full-sample weights.
If \code{FALSE}, compute variances from sums of squares around the mean estimate from the replicate weights.
For Fay's generalized replication method, setting \code{mse = FALSE} can potentially
lead to large underestimates of variance.}

\item{compress}{This reduces the computer memory required to represent the replicate weights and has no
impact on estimates.}
}
\value{
A replicate design object, with class \code{svyrep.design}, which can be used with the usual functions,
such as \code{svymean()} or \code{svyglm()}.

Use \code{weights(..., type = 'analysis')} to extract the matrix of replicate weights.

Use \code{as_data_frame_with_weights()} to convert the design object to a data frame with columns
for the full-sample and replicate weights.
}
\description{
Converts a survey design object to a replicate design object
with replicate weights formed using the generalized replication method of Fay (1989).
The generalized replication method forms replicate weights
from a textbook variance estimator, provided that the variance estimator
can be represented as a quadratic form whose matrix is positive semidefinite
(this covers a large class of variance estimators).
}
\section{Statistical Details}{

See Fay (1989) for a full description of this replication method,
or see the documentation in \link[svrep]{make_fays_gen_rep_factors} for implementation details.

See \link[svrep]{variance-estimators} for a
description of each variance estimator available for use with
this function.

Use \code{\link[svrep]{rescale_reps}} to eliminate negative adjustment factors.
}

\section{Two-Phase Designs}{

For a two-phase design, \code{variance_estimator} should be a list of variance estimators' names,
with two elements, such as \code{list('Ultimate Cluster', 'Poisson Horvitz-Thompson')}.
In two-phase designs, only the following estimators may be used for the second phase:
\itemize{
  \item "Ultimate Cluster"
  \item "Stratified Multistage SRS"
  \item "Poisson Horvitz-Thompson"
}
For statistical details on the handling of two-phase designs,
see the documentation for \link[svrep]{make_twophase_quad_form}.
}

\examples{
if (FALSE) {

  library(survey)

  ## Load an example systematic sample ----
  data('library_stsys_sample', package = 'svrep')

  ## First, ensure data are sorted in same order as was used in sampling
  library_stsys_sample <- library_stsys_sample |>
    sort_by(~ SAMPLING_SORT_ORDER)

  ## Create a survey design object
  design_obj <- svydesign(
    data = library_stsys_sample,
    strata = ~ SAMPLING_STRATUM,
    ids = ~ 1,
    fpc = ~ STRATUM_POP_SIZE
  )

  ## Convert to generalized replicate design

  gen_rep_design_sd2 <- as_fays_gen_rep_design(
    design = design_obj,
    variance_estimator = "SD2",
    max_replicates = 250,
    mse = TRUE
  )

  svytotal(x = ~ TOTSTAFF, na.rm = TRUE, design = gen_rep_design_sd2)
}
}
\references{
The generalized replication method was first proposed in
Fay (1984). Fay (1989) refined the generalized replication method
to produce "balanced" replicates, in the sense that
each replicate contributes equally to variance estimates.
The advantage of balanced replicates is that one can
still obtain a reasonable variance estimate
by using only a random subset of the replicates.

- Ash, S. (2014). "\emph{Using successive difference replication for estimating variances}."
\strong{Survey Methodology}, Statistics Canada, 40(1), 47-59.
\cr \cr
- Beaumont, J.-F.; Émond, N. (2022).
"\emph{A Bootstrap Variance Estimation Method for Multistage Sampling and Two-Phase Sampling When Poisson Sampling Is Used at the Second Phase}."
\strong{Stats}, \emph{5}: 339-357.
https://doi.org/10.3390/stats5020019
\cr \cr
- Breidt, F. J., Opsomer, J. D., & Sanchez-Borrego, I. (2016). 
"\emph{Nonparametric Variance Estimation Under Fine Stratification: An Alternative to Collapsed Strata}." 
\strong{Journal of the American Statistical Association}, 111(514), 822-833. https://doi.org/10.1080/01621459.2015.1058264
\cr \cr
- Deville, J. C., and Tillé, Y. (2005). "\emph{Variance approximation under balanced sampling.}"
\strong{Journal of Statistical Planning and Inference}, 128, 569-591.
\cr \cr
- Dippo, Cathryn, Robert Fay, and David Morganstein. 1984. "Computing Variances from Complex Samples with Replicate Weights." In, 489-94. Alexandria, VA: American Statistical Association. http://www.asasrms.org/Proceedings/papers/1984_094.pdf.
\cr \cr
- Fay, Robert. 1984. "Some Properties of Estimates of Variance Based on Replication Methods." In, 495-500. Alexandria, VA: American Statistical Association. http://www.asasrms.org/Proceedings/papers/1984_095.pdf.
\cr \cr
- Fay, Robert. 1989. "Theory And Application Of Replicate Weighting For Variance Calculations." In, 495-500. Alexandria, VA: American Statistical Association. http://www.asasrms.org/Proceedings/papers/1989_033.pdf
\cr \cr
- Matei, Alina, and Yves Tillé. (2005).
"\emph{Evaluation of Variance Approximations and Estimators
in Maximum Entropy Sampling with Unequal Probability and Fixed Sample Size.}"
\strong{Journal of Official Statistics}, 21(4):543-70.
}
\seealso{
For greater customization of the method, \code{\link[svrep]{make_quad_form_matrix}} can be used to
represent several common variance estimators as a quadratic form's matrix,
which can then be used as an input to \code{\link[svrep]{make_fays_gen_rep_factors}}.
}
