% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aliases.R
\name{pipe_add}
\alias{pipe_add}
\title{Add pipeline step}
\usage{
pipe_add(
  pip,
  step,
  fun,
  params = list(),
  description = "",
  group = step,
  keepOut = FALSE
)
}
\arguments{
\item{pip}{\code{Pipeline} object}

\item{step}{\code{string} the name of the step. Each step name must
be unique.}

\item{fun}{\code{function} or name of the function to be applied at
the step. Both existing and anonymous/lambda functions can be used.
All function parameters must have default values. If a parameter
is missing a default value in the function signature, alternatively,
it can be set via the \code{params} argument (see Examples section with
\code{\link[=mean]{mean()}} function).}

\item{params}{\code{list} list of parameters to set or overwrite
parameters of the passed function.}

\item{description}{\code{string} optional description of the step}

\item{group}{\code{string} output collected after pipeline execution
(see \code{\link[=pipe_collect_out]{pipe_collect_out()}} is grouped by the defined group
names. By default, this is the name of the step, which comes in
handy when the pipeline is copy-appended multiple times to keep
the results of the same function/step grouped at one place.}

\item{keepOut}{\code{logical} if \code{FALSE} (default) the output of the
step is not collected when calling \code{\link[=pipe_collect_out]{pipe_collect_out()}} after the pipeline
run. This option is used to only keep the results that matter
and skip intermediate results that are not needed. See also
function \code{\link[=pipe_collect_out]{pipe_collect_out()}} for more details.}
}
\value{
returns the \code{Pipeline} object invisibly
}
\description{
A pipeline consists of a series of steps, which usually
are added one by one. Each step is made up of a function computing
something once the pipeline is run. This function can be an existing
R function (e.g. \code{\link[=mean]{mean()}}) or an anonymous/lambda function specifically
defined for the pipeline. One useful feature is that function
parameters can refer to results of earlier pipeline steps using the
syntax \code{x = ~earlier_step_name} - see the Examples for more details.
}
\examples{
# Add steps with lambda functions
p <- pipe_new("myPipe", data = 1)
pipe_add(p, "s1", \(x = ~data) 2*x)  # use input data
pipe_add(p, "s2", \(x = ~data, y = ~s1) x * y)
try(pipe_add(p, "s2", \(z = 3) 3)) # error: step 's2' exists already
try(pipe_add(p, "s3", \(z = ~foo) 3)) # dependency 'foo' not found
p

# Add step with existing function
p <- pipe_new("myPipe", data = c(1, 2, NA, 3, 4))
try(pipe_add(p, "calc_mean", mean))  # default value for x is missing
pipe_add(p, "calc_mean", mean, params = list(x = ~data, na.rm = TRUE))
p |> pipe_run() |> pipe_get_out("calc_mean")

# Step description
p <- pipe_new("myPipe", data = 1:10)
pipe_add(p, "s1", \(x = ~data) 2*x, description = "multiply by 2")
print(p, verbose = TRUE) # print all columns including description


# Group output
p <- pipe_new("myPipe", data = data.frame(x = 1:2, y = 3:4))
pipe_add(p, "prep_x", \(data = ~data) data$x, group = "prep")
pipe_add(p, "prep_y", \(data = ~data) (data$y)^2, group = "prep")
pipe_add(p, "sum", \(x = ~prep_x, y = ~prep_y) x + y)
p |> pipe_run() |> pipe_collect_out(all = TRUE)
}
