#' Plot Results from the Pettitt Change‐Point Test
#'
#' Creates a two‐panel visualization of the Mann–Whitney–Pettitt test. The 
#' upper panel plots the Pettitt \eqn{U_t} statistic over time along with the 
#' significance threshold and potential change point. The lower panel displays 
#' the annual maximum series data with an optional trend line, the period 
#' mean(s), and potential change point(s).
#'
#' @param results A list generated by [eda_pettitt_test()].
#'
#' @param show_line If `TRUE` (default), draw a fitted line through the data.
#'
#' @param ... Optional named arguments: 'title', 'top_xlabel', 'top_ylabel', 
#'   'bottom_xlabel' and 'bottom_ylabel'.
#'
#' @return A \code{patchwork} object with two \code{ggplot2} panels stacked vertically.
#'
#' @examples
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' years <- seq(from = 1901, to = 2000)
#' results <- eda_pettitt_test(data, years)
#' plot_pettitt_test(results)
#'
#' @seealso [eda_pettitt_test()]
#'
#' @import ggplot2
#' @import patchwork
#' @export

plot_pettitt_test <- function(results, show_line = TRUE, ...) {

	change_df <- results$change_points

	# Add u_series to df and create change point df (if change point is significant)
	df <- data.frame(
		value = results$data,
		year = results$years,
		u_series = results$u_series
	)

	# Compute the segment means for the plot
	n <- length(results$u_series)
	ends <- if (nrow(change_df) == 0) c(1, n) else c(1, change_df$index, n) 
	get_segment_mean <- function(i) mean(df$value[ends[i]:ends[i + 1]])

	segment_df <- data.frame(
		x = df$year[ends[-length(ends)]],            
		xend = df$year[ends[-1]],
		y = sapply(1:(length(ends) - 1), get_segment_mean)
	)

	# Capture optional arguments
	args <- list(...)

	# Define labels for the plot 
    title <- args$title %||% "Pettitt Test"
	top_xlabel <- args$top_xlabel %||% "Year" 
	top_ylabel <- args$top_ylabel %||% expression(U[t] ~ Statistic)
	bottom_xlabel <- args$bottom_xlabel %||% "Year" 
	bottom_ylabel <- args$bottom_ylabel %||% expression(Streamflow ~ m^3/s)

	# First subplot: Mann-Whitney-Pettitt Test
	p1 <- ggplot(df, aes(x = .data$year, y = .data$u_series)) +
		geom_line(aes(color = "black"), linewidth = 1) +
		geom_hline(
			aes(yintercept = results$bound, color = "red"),
			linewidth = 1.2,
			linetype = "dashed",
		) +
		geom_point(
			data = change_df,
			aes(y = .data$statistic, color = "blue"), 
			size = 4
		) +
		labs(title = title, x = top_xlabel, y = top_ylabel, color = "Legend") + 
		scale_color_manual(
			values = c("black" = "black", "red" = "red", "blue" = "blue"),
			breaks = c("black", "red", "blue"),
			labels = c(top_ylabel, "Change Point Threshold", "Potential Change Point")
		)
		
	# Also plot the original flow data and segment means
	p2 <- ggplot(df, aes(x = .data$year, y = .data$value)) + 
		geom_point(aes(color = "dimgray"), size = 2.25) +
		(if (show_line) geom_line(color = "dimgray", linewidth = 1.1) else NULL) + 
		geom_segment(
			data = segment_df, 
			aes(x = .data$x, xend = .data$xend, y = .data$y, color = "black"),
			linewidth = 1.2 
		) +
		geom_point(data = change_df, aes(color = "blue"), size = 4) +
		labs(x = bottom_xlabel, y = bottom_ylabel, color = "Legend") +
		scale_color_manual(
			values = c("dimgray" = "dimgray", "black" = "black", "blue" = "blue"),
			breaks = c("dimgray", "black", "blue"),
			labels = c(bottom_ylabel, "Segment Mean(s)", "Potential Change Point")
		)	

	# Stack plots on top of each other and return
	(add_theme(add_scales(p1)) / add_theme(add_scales(p2))) + plot_annotation(title = "")

}
