% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_sub.R
\name{add_sub}
\alias{add_sub}
\title{Add annotation underneath a plot}
\usage{
add_sub(plot, label, x = 0.5, y = 0.5, hjust = 0.5, vjust = 0.5,
  vpadding = grid::unit(1, "lines"), fontfamily = "", fontface = "plain",
  colour = "black", size = 14, angle = 0, lineheight = 0.9)
}
\arguments{
\item{plot}{A ggplot object or gtable object derived from a ggplot object.}

\item{label}{The label with which the plot should be annotated. Can be a plotmath expression.}

\item{x}{The x position of the label}

\item{y}{The y position of the label}

\item{hjust}{Horizontal justification}

\item{vjust}{Vertical justification}

\item{vpadding}{Vertical padding. The total vertical space added to the label, given in grid
units. By default, this is added equally above and below the label. However, by changing the
y and vjust parameters, this can be changed.}

\item{fontfamily}{The font family}

\item{fontface}{The font face ("plain", "bold", etc.)}

\item{colour}{Text color}

\item{size}{Point size of text}

\item{angle}{Angle at which text is drawn}

\item{lineheight}{Line height of text}
}
\value{
A gtable object holding the modified plot.
}
\description{
This function can add an arbitrary label or mathematical expression underneath
the plot, similar to the \code{sub} parameter in base R. It is mostly superseded now by the
\code{caption} argument to \code{\link[ggplot2:labs]{ggplot2::labs()}}, and it is recommended to use \code{caption} instead of
\code{add_sub()} whenever possible.
}
\details{
The exact location where the
label is placed is controlled by the parameters \code{x}, \code{y}, \code{hjust}, and
\code{vjust}. By default, all these parameters are set to 0.5, which places the label
centered underneath the plot panel. A value of \code{x = 0} indicates the left boundary
of the plot panel and a value of \code{x = 1} indicates the right boundary. The parameter
\code{hjust} works just as elsewhere in ggplot2. Thus, \code{x = 0, hjust = 0} places
the label left-justified at the left boundary of the plot panel, \code{x = 0.5, hjust = 0.5}
places the label centered underneath the plot panel, and \code{x = 1, hjust = 1} places
it right-justified at the right boundary of the plot panel. \code{x}-values below 0 or
above 1 are allowed, and they move the label beyond the limits of the plot panel.

The \code{y} coordinates are relative to the added vertical space that is introduced
underneath the x-axis label to place the annotation. A value of \code{y=0} indicates
the bottom-most edge of that space and a value of \code{y=1} indicates the top-most
edge of that space. The total height of the added space is given by the height needed
to draw the label plus the value of \code{vpadding}. Thus, if \code{y=0, vjust=0} then
the extra padding is added entirely above the label, if \code{y=1, vjust=1} then the
extra padding is added entirely below the label, and if \code{y=0.5, vjust=0.5} (the
default) then the extra padding is added equally above and below the label. As is the
case with \code{x}, \code{y}-values outside the range 0-1 are allowed. In particular,
for sufficiently large values of \code{y}, the label will eventually be located inside
the plot panel.
}
\examples{
p1 <- ggplot(mtcars, aes(mpg, disp)) + geom_line(colour = "blue") + background_grid(minor='none')
ggdraw(add_sub(p1, "This is an annotation.\\nAnnotations can span multiple lines."))

# You can also do this repeatedly.
p2 <- add_sub(p1, "This formula has no relevance here:", y  = 0, vjust = 0)
p3 <- add_sub(p2, expression(paste(a^2+b^2, " = ", c^2)))
ggdraw(p3)

#This code also works with faceted plots:
plot.iris <- ggplot(iris, aes(Sepal.Length, Sepal.Width)) +
  geom_point() + facet_grid(. ~ Species) + stat_smooth(method = "lm") +
  background_grid(major = 'y', minor = "none") + # add thin horizontal lines
  panel_border() # and a border around each panel
p2 <- add_sub(plot.iris, "Annotation underneath a faceted plot, left justified.", x = 0, hjust = 0)
ggdraw(p2)

# Finally, it is possible to move the annotation inside of the plot if desired.
ggdraw(add_sub(p1, "Annotation inside plot", vpadding=grid::unit(0, "lines"),
       y = 6, x = 0.03, hjust = 0))
}
