% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IGMM.R
\name{IGMM}
\alias{IGMM}
\title{Iterative Generalized Method of Moments -- IGMM}
\usage{
IGMM(
  y,
  type = c("h", "hh", "s"),
  skewness.x = 0,
  kurtosis.x = 3,
  tau.init = get_initial_tau(y, type),
  robust = FALSE,
  tol = .Machine$double.eps^0.25,
  location.family = TRUE,
  not.negative = NULL,
  max.iter = 100,
  delta.lower = -1,
  delta.upper = 3
)
}
\arguments{
\item{y}{a numeric vector of real values.}

\item{type}{type of Lambert W \eqn{\times} F distribution: skewed \code{"s"};
heavy-tail \code{"h"}; or skewed heavy-tail \code{"hh"}.}

\item{skewness.x}{theoretical skewness of input X; default \code{0}
(symmetric distribution).}

\item{kurtosis.x}{theoretical kurtosis of input X; default \code{3} (Normal
distribution reference).}

\item{tau.init}{starting values for IGMM algorithm; default:
\code{\link{get_initial_tau}}.  See also \code{\link{gamma_Taylor}} and
\code{\link{delta_Taylor}}.}

\item{robust}{logical; only used for \code{type = "s"}. If \code{TRUE} a
robust estimate of asymmetry is used (see
\code{\link{medcouple_estimator}}); default: \code{FALSE}.}

\item{tol}{a positive scalar specifiying the tolerance level for terminating
the iterative algorithm. Default: \code{.Machine$double.eps^0.25}}

\item{location.family}{logical; tell the algorithm whether the underlying
input should have a location family distribution (for example, Gaussian
input); default: \code{TRUE}. If \code{FALSE} (e.g., for
\code{"exp"}onential input), then \code{tau['mu_x'] = 0} throughout the
optimization.}

\item{not.negative}{logical; if \code{TRUE}, the estimate for \eqn{\gamma} or
\eqn{\delta} is restricted to non-negative reals. If it is set to
\code{NULL} (default) then it will be set internally to \code{TRUE} for
heavy-tail(s) Lambert W\eqn{ \times} F distributions (\code{type = "h"}
or \code{"hh"}).  For skewed Lambert W\eqn{ \times} F (\code{type = "s"})
it will be set to \code{FALSE}, unless it is not a location-scale family
(see \code{\link{get_distname_family}}).}

\item{max.iter}{maximum number of iterations; default: \code{100}.}

\item{delta.lower, delta.upper}{lower and upper bound for
\code{\link{delta_GMM}} optimization.  By default: \code{-1} and \code{3}
which covers most real-world heavy-tail scenarios.}
}
\value{
A list of class \code{LambertW_fit}: 
\item{tol}{see Arguments} 
\item{data}{ data \code{y}}
\item{n}{ number of observations} 
\item{type}{see Arguments} 
\item{tau.init}{ starting values for \eqn{\tau} } 
\item{tau}{ IGMM estimate for \eqn{\tau} }
\item{tau.trace}{entire iteration trace of \eqn{\tau^{(k)}}, \eqn{k = 0, ..., K}, where 
\code{K <= max.iter}.}
\item{sub.iterations}{number of iterations only performed in GMM algorithm to find optimal \eqn{\gamma} (or \eqn{\delta})} 
\item{iterations}{number of iterations to update \eqn{\mu_x} and
\eqn{\sigma_x}. See References for detals.} 
\item{hessian}{ Hessian matrix (obtained from simulations; see References)} 
\item{call}{function call}
\item{skewness.x, kurtosis.x}{ see Arguments} 
\item{distname}{ a character string describing distribution characteristics given
the target theoretical skewness/kurtosis for the input. Same information as \code{skewness.x} and \code{kurtosis.x} but human-readable.} 
\item{location.family}{see Arguments} 
\item{message}{message from the optimization method. What kind of convergence?} 
\item{method}{estimation method; here: \code{"IGMM"}}
}
\description{
An iterative method of moments estimator to find this \eqn{\tau = (\mu_x,
    \sigma_x, \gamma)} for \code{type = 's'} (\eqn{\tau = (\mu_x, \sigma_x,
    \delta)} for \code{type = 'h'} or \eqn{\tau = (\mu_x, \sigma_x, \delta_l,
    \delta_r)} for \code{type = "hh"}) which minimizes the distance between
    the sample and theoretical skewness (or kurtosis) of \eqn{\boldsymbol x}
    and X.

This algorithm is only well-defined for data with finite mean and variance
    input X.  See \code{\link{analyze_convergence}} and references therein
    for details.
}
\details{
For algorithm details see the References.
}
\examples{

# estimate tau for the skewed version of a Normal
y <- rLambertW(n = 100, theta = list(beta = c(2, 1), gamma = 0.2), 
               distname = "normal")
fity <- IGMM(y, type = "s")
fity
summary(fity)
plot(fity)
\dontrun{
# estimate tau for the skewed version of an exponential
y <- rLambertW(n = 100, theta = list(beta = 1, gamma = 0.5), 
               distname = "exp")
fity <- IGMM(y, type = "s", skewness.x = 2, location.family = FALSE)
fity
summary(fity)
plot(fity)

# estimate theta for the heavy-tailed version of a Normal = Tukey's h
y <- rLambertW(n = 100, theta = list(beta = c(2, 1), delta = 0.2), 
               distname = "normal")
system.time(
fity <- IGMM(y, type = "h")
)
fity
summary(fity)
plot(fity)
}
}
\seealso{
\code{\link{delta_GMM}}, \code{\link{gamma_GMM}}, \code{\link{analyze_convergence}}
}
\author{
Georg M. Goerg
}
\keyword{iteration}
\keyword{optimize}
